#if NUNIT
using System;
using System.Data;
using NUnit.Framework;

using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.DAL.HL7.Parsers;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary.Messages;
using gov.va.med.vbecs.DAL.HL7AL;

using gov.va.med.vbecs.UnitTests;

namespace gov.va.med.vbecs.DAL.HL7.UnitTests.Parsers
{
	[TestFixture]
    public class Parsers_PatientMergeHL7Parser : BaseTest 
	{
		private const string CARRIAGE_RETURN = "\x0D";

		/// <summary>
		/// Test Merge Message
		/// </summary>
		private const string PATIENT_MERGE_MESSAGE =
			"MSH^~|\\&^MPIF TRIGGER^^VBECS ADT^589^20110410173117-0500^^ADT~A40^5894298240^T^2.4^^^AL^AL^" +
			CARRIAGE_RETURN +
			"EVN^^^^^53342~USER~TEST~~~~~~USVHA&&0363~L~~~NI~VA FACILITY ID&589&L^^589" +
			CARRIAGE_RETURN +
			"PID^1^1000045036V498441~~~USVHA&&0363~NI~VA FACILITY ID&200M&L^1000045036V498441~~~USVHA&&0363~NI~VA FACILITY ID&200M&L|321000004~~~USSSA&&0363~SS~VA FACILITY ID&589&L|412583~~~USVHA&&0363~PI~VA FACILITY ID&589&L|321000003~~~USSSA&&0363~SS~VA FACILITY ID&589&L~~20110410|321000002~~~USSSA&&0363~SS~VA FACILITY ID&589&L~~20110410|5890045036V502660~~~USVHA&&0363~NI~VA FACILITY ID&589&L~~20061213|5890045035V840731~~~USVHA&&0363~NI~VA FACILITY ID&589&L~~20061213|1000045035V840731~~~USVHA&&0363~NI~VA FACILITY ID&200M&L~~20061213|5890045034V975882~~~USVHA&&0363~NI~VA FACILITY ID&589&L~~20061213|1000045034V103607~~~USVHA&&0363~NI~VA FACILITY ID&200M&L~~20110310^^MERGEN~PATIENT~~~~~L|MERGEM~PATIENT~~~~~A|MERGEL~PATIENT~~~~~A^DOFKDP~~~~~~M^19230101^F^^\"\"^22 WEST ST~\"\"~MUNDELEIN~IL~60060~USA~P~\"\"~097|~~CHICAGO~IL~~~N^097^(630)449-8332~PRN~PH|(630)456-8484~WPN~PH^(630)456-8484^^\"\"^0^^321000004^^^\"\"^CHICAGO IL^N^^^^^\"\"^^" +
			CARRIAGE_RETURN +
			"PD1^^^VA HEARTLAND - WEST, VISN 15~D~589" +
			CARRIAGE_RETURN +
			// MRG segment tests invalid "MERGING INTO" name fix
			"MRG^1000046402V857601~~~USVHA&&0363~NI~VA FACILITY ID&589&L|412727~~~USVHA&&0363~PI~VA FACILITY ID&589&L^^^^^^ENTRY~MERGING INTO 412728 USE~THAT~~~~L" +
			CARRIAGE_RETURN;

		/// <summary>
		/// Used for negative testing
		/// </summary>
		private const string CPRS_MESSAGE = 
			"MSH|^~\\&|OERR|589|VBECS|589|20080526135401-0500||OMG^O19|5894312166|T|2.4|||AL|AL|USA" +
			CARRIAGE_RETURN +
			"PID|||5890045694V193464^^^^NI~493013508^^^^SS~14161^^^^PI||HALZ^ALUUN^J||19601126|M" +
			CARRIAGE_RETURN +
			"PV1||O|KC-1010 NEW PT CLINIC 2|||||||" + 
			CARRIAGE_RETURN +
			"ORC|NW|22179410^OR||22179406^OR|||||200805261354-0500|53342^USER^ONE||53342^USER^ONE||||^Transfuse|589^VA HEARTLAND - WEST, VISN 15|||" + 
			CARRIAGE_RETURN +
			"OBR|1|22179410^OR||1^TYPE \\T\\ SCREEN^99VBC^^|||||||O||89932;3080526;4;6478||^^SEND PATIENT||||||||||||^^^20080526^200805261354-0500^R^^" + 
			CARRIAGE_RETURN +
			"OBX|1||191^ALANINE AMINOTRANSFERASE^99LRT||17|IU/L|8-40||||F|||20020114093059-0500" +
			CARRIAGE_RETURN +
			"DG1|1||^^^^^|||A " +
			CARRIAGE_RETURN;
		
		private HL7Interface _patientMergeInterface;

		private PatientMergeHL7Parser _parser;

		[SetUp]
		public void SetUp()
		{
			_patientMergeInterface = new HL7Interface( InterfaceName.Patient_Merge.ToString() );
			//
			_parser = new PatientMergeHL7Parser();
		}

		/// <summary>
		/// ParseHL7Message - PASS
		/// </summary>
		[Test]
		public void ParseHL7Message_Pass()
		{
			HL7ProtocolMessage message = _parser.ParseHL7Message( _patientMergeInterface, PATIENT_MERGE_MESSAGE ); 
			Assert.IsNotNull( message, "Merge Message Parsed" );
			Assert.AreEqual("AA", HL7Utility.ParseGetAckCode(message.GetMessage()), "Ack Check" );
		}

		/// <summary>
		/// ParseHL7Message - FAIL (Message is NULL)
		/// </summary>
		[Test]
		[ExpectedException( typeof( ArgumentNullException) )]
		public void ParseHL7Message_Fail_Null_Message()
		{
			_parser.ParseHL7Message( _patientMergeInterface, null );
		}

		/// <summary>
		/// ParseHL7Message - FAIL (Interface is NULL)
		/// </summary>
		[Test]
		[ExpectedException( typeof( ArgumentNullException ) )]
		public void ParseHL7Message_Null_Interface()
		{
			_parser.ParseHL7Message( null, PATIENT_MERGE_MESSAGE );
		}

		/// <summary>
		/// ParseHL7Message - FAIL (Interface is wrong)
		/// </summary>
		[Test]
		[ExpectedException( typeof( HL7Exception ) )]
		public void ParseHL7Message_Wrong_Interface()
		{
			_parser.ParseHL7Message( new HL7Interface(InterfaceName.CPRS.ToString()), PATIENT_MERGE_MESSAGE );
		}

		/// <summary>
		/// ParseHL7Message - FAIL (Message Type is wrong)
		/// </summary>
		[Test]
		[ExpectedException( typeof( HL7Exception ) )]
		public void ParseHL7Message_Fail_Wrong_Message_Type()
		{
			_parser.ParseHL7Message( _patientMergeInterface, CPRS_MESSAGE );
		}
	}
}
#endif